﻿using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Net.Http;
using System.Runtime.Serialization;
using System.Threading.Tasks;

namespace NISLSCWebClient
{
   public class RequestParams
   {
      public string access { get; set; }
      public string[] devices { get; set; }
      public bool force_reserve { get; set; }
   }

   public class InitializeSessionRequest
   {
      public string id { get; set; }
      public string jsonrpc { get; set; }
      public string method { get; set; }

      public RequestParams @params { get; set; }
   }

   public class InitializeSessionResult
   {
      public string session_id { get; set; }
   }

   public class InitializeSessionResponse
   {
      public string id { get; set; }
      public string jsonrpc { get; set; }
      public InitializeSessionResult result { get; set; }
   }

   public class ErrorData
   {
      public decimal nislsc_code { get; set; }
      public string nislsc_message { get; set; }
   }

   public class ErrorResult
   {
      public decimal code { get; set; }
      public string message { get; set; }
      public ErrorData data { get; set; }
   }

   [Serializable]
   public class ErrorResponse : SystemException
   {
      public string id { get; set; }
      public string jsonrpc { get; set; }
      public ErrorResult error { get; set; }
      public ErrorResponse(SerializationInfo info, StreamingContext context)
      {
         if (info != null)
         {
            this.id = info.GetString("id");
            this.jsonrpc = info.GetString("jsonrpc");
            this.error = (ErrorResult)info.GetValue("error", typeof(ErrorResult));
         }
      }
      public override void GetObjectData(SerializationInfo info, StreamingContext context)
      {
         base.GetObjectData(info, context);

         if (info != null)
         {
            info.AddValue("id", this.id);
            info.AddValue("jsonrpc", this.jsonrpc);
            info.AddValue("error", this.error);
         }
      }
   }

   class Program
   {
      static HttpClient client = new HttpClient();

      static async Task<InitializeSessionResponse> InitSessionAsync(InitializeSessionRequest request)
      {
         HttpResponseMessage response = await client.PostAsJsonAsync(
             "call", request);
         if (!response.IsSuccessStatusCode)
         {
            throw new Exception("Status Error code: " + (int)response.StatusCode);
         }
         else
         {
            String content = await response.Content.ReadAsStringAsync();
            JObject jsonContent = JObject.Parse(content);
            if (jsonContent["error"] != null)
            {
               throw JsonConvert.DeserializeObject<ErrorResponse>(content);
            }
            return JsonConvert.DeserializeObject<InitializeSessionResponse>(content);
         }
      }

      static async Task RunAsync()
      {
         client.BaseAddress = new Uri("http://10.144.160.151/nislsc/");
         try
         {
            // Create a new request
            InitializeSessionRequest request = new InitializeSessionRequest
            {
               id = "1",
               jsonrpc = "2.0",
               method = "initializeSession",
               @params = new RequestParams
               {
                  access = "ReadWrite",
                  force_reserve = true,
                  devices = new string[1] { "SLSC-12001-030FD4DE-Mod8" }
               }
            };

            //Get response
            var response = await InitSessionAsync(request);
            Console.WriteLine("session id = " + response.result.session_id);

         }
         catch (ErrorResponse e)
         {
            if (e.error.data != null)
            {
               Console.WriteLine("NI-SLSC error message: " + e.error.data.nislsc_message);
            }
            else
            {
               Console.WriteLine("JSON-RPC error message:" + e.error.message);
            }
         }
         catch (Exception e)
         {
            Console.WriteLine(e.Message);
         }
      }

      static void Main(string[] args)
      {
         RunAsync().GetAwaiter().GetResult();
      }
   }
}
